<?php

namespace App\Traits;

trait HasTranslations
{
    /**
     * Get a translation for a given field and locale.
     * Falls back to: requested locale → other locale → original column value.
     */
    public function getTranslation(string $field, ?string $locale = null): ?string
    {
        $locale = $locale ?? app()->getLocale();
        $translationsField = $field . '_translations';

        // Try getting from translations JSON column
        $translations = $this->getAttribute($translationsField);

        if (is_string($translations)) {
            $translations = json_decode($translations, true);
        }

        if (is_array($translations)) {
            // Return requested locale
            if (!empty($translations[$locale])) {
                return $translations[$locale];
            }

            // Fallback to other locale
            $fallback = $locale === 'ar' ? 'en' : 'ar';
            if (!empty($translations[$fallback])) {
                return $translations[$fallback];
            }
        }

        // Final fallback: original column
        return $this->getAttribute($field);
    }

    /**
     * Set a translation for a given field and locale.
     */
    public function setTranslation(string $field, string $locale, ?string $value): self
    {
        $translationsField = $field . '_translations';
        $translations = $this->getAttribute($translationsField) ?? [];

        if (is_string($translations)) {
            $translations = json_decode($translations, true) ?? [];
        }

        $translations[$locale] = $value;
        $this->setAttribute($translationsField, $translations);

        return $this;
    }

    /**
     * Get all translations for a field.
     */
    public function getTranslations(string $field): array
    {
        $translationsField = $field . '_translations';
        $translations = $this->getAttribute($translationsField);

        if (is_string($translations)) {
            return json_decode($translations, true) ?? [];
        }

        return is_array($translations) ? $translations : [];
    }

    /**
     * Get the translated name (convenience accessor).
     */
    public function getTranslatedNameAttribute(): string
    {
        return $this->getTranslation('name') ?? $this->name ?? '';
    }

    /**
     * Get the translated description (convenience accessor).
     */
    public function getTranslatedDescriptionAttribute(): ?string
    {
        return $this->getTranslation('description');
    }

    /**
     * Get the translated short description (convenience accessor).
     */
    public function getTranslatedShortDescriptionAttribute(): ?string
    {
        return $this->getTranslation('short_description');
    }
}
