<?php

/**
 * Polyfill for PHP intl extension
 * Loads before vendor autoload to prevent Laravel Number class errors
 */

// Override Laravel's Number class for systems without intl extension
if (!extension_loaded('intl')) {
    
    // Polyfill for locale_get_display_name used by Filament Translatable Plugin
    if (!function_exists('locale_get_display_name')) {
        function locale_get_display_name($locale, $in_locale = null) {
            $map = [
                'en' => 'English',
                'ar' => 'العربية',
            ];
            return $map[strtolower(substr($locale, 0, 2))] ?? $locale;
        }
    }
    
    // Define NumberFormatter constants
    if (!defined('NUMBER_FORMATTER_DECIMAL')) {
        define('NUMBER_FORMATTER_DECIMAL', 1);
    }
    
    // Define NumberFormatter class if not exists
    if (!class_exists('NumberFormatter')) {
        class NumberFormatter
        {
            const DECIMAL = 1;
            const CURRENCY = 2;
            const PERCENT = 3;
            const SCIENTIFIC = 4;
            const SPELLOUT = 5;
            const ORDINAL = 6;
            const DURATION = 7;
            const PATTERN_RULEBASED = 9;
            const IGNORE = 0;
            const DEFAULT_STYLE = 1;
            const FRACTION_DIGITS = 8;
            const MAX_FRACTION_DIGITS = 9;
            const MIN_FRACTION_DIGITS = 10;
            const INTEGER_DIGITS = 11;
            const GROUPING_USED = 1;
            const DECIMAL_ALWAYS_SHOWN = 2;
            const TYPE_DEFAULT = 0;
            const TYPE_INT32 = 1;
            const TYPE_INT64 = 2;
            const TYPE_DOUBLE = 3;
            const TYPE_CURRENCY = 4;
            const PAD_BEFORE_PREFIX = 0;
            const PAD_AFTER_PREFIX = 1;
            const PAD_BEFORE_SUFFIX = 2;
            const PAD_AFTER_SUFFIX = 3;
            const PARSE_INT_ONLY = 0;
            const GROUPING_SIZE = 3;
            const ROUNDING_MODE = 4;
            const ROUNDING_INCREMENT = 5;
            const FORMAT_WIDTH = 6;
            const PADDING_CHARACTER = 7;
            const CURRENCY_CODE = 18;
            const DEFAULT_RULESET = 19;
            
            private $locale;
            private $style;
            private $attributes = [];
            private $textAttributes = [];
            private $pattern;
            
            public function __construct($locale, $style = self::DECIMAL, $pattern = null)
            {
                $this->locale = $locale;
                $this->style = $style;
                $this->pattern = $pattern;
            }
            
            public function setAttribute($attr, $value)
            {
                $this->attributes[$attr] = $value;
            }
            
            public function getAttribute($attr)
            {
                return $this->attributes[$attr] ?? 0;
            }
            
            public function setTextAttribute($attr, $value)
            {
                $this->textAttributes[$attr] = $value;
            }
            
            public function getTextAttribute($attr)
            {
                return $this->textAttributes[$attr] ?? '';
            }
            
            public function format($number)
            {
                switch ($this->style) {
                    case self::PERCENT:
                        return number_format($number * 100, $this->getFractionDigits()) . '%';
                    case self::SPELLOUT:
                        return (string) $number;
                    case self::ORDINAL:
                        return (string) $number;
                    case self::CURRENCY:
                    case self::DECIMAL:
                    default:
                        return number_format($number, $this->getFractionDigits());
                }
            }
            
            public function formatCurrency($amount, $currency)
            {
                $decimals = $this->getFractionDigits();
                return $currency . ' ' . number_format($amount, $decimals);
            }
            
            public function parse($string, $type = self::TYPE_DOUBLE)
            {
                $string = preg_replace('/[^0-9.\-]/', '', $string);
                return $type == self::TYPE_INT32 ? (int) $string : (float) $string;
            }
            
            public function parseCurrency($string, &$currency, &$position = null)
            {
                $currency = '';
                $position = 0;
                return $this->parse($string);
            }
            
            private function getFractionDigits()
            {
                return $this->attributes[self::FRACTION_DIGITS] ?? 
                       ($this->attributes[self::MAX_FRACTION_DIGITS] ?? 0);
            }
        }
    }
    
    // Override Laravel's Number class
    if (!class_exists('Illuminate\Support\Number')) {
        require_once __DIR__ . '/../app/Support/Number.php';
    }
}
