@props(['data' => []])

@php

    $title = $data['title'] ?? "";
    $description = $data['description'] ?? "";
    $packages_source = $data['packages_source'] ?? 'featured';
    $limit = $data['limit'] ?? 6;
    $columns = $data['columns'] ?? '3';
    $show_view_all = $data['show_view_all'] ?? true;
    $view_all_text = $data['view_all_text'] ?? "";
    $view_all_url = $data['view_all_url'] ?? "";
    
    // Fetch packages based on source
    $query = \App\Models\Package::query()->active();
    
    switch($packages_source) {
        case 'latest':
            $query->latest();
            break;
        case 'featured':
            $query->featured()->orderBy('sort_order');
            break;
        case 'all':
            $query->orderBy('sort_order');
            break;
        default:
            $query->orderBy('sort_order');
    }
    
    $packages = $query->limit($limit)->get();
    
    $gridClass = match($columns) {
        '2' => 'grid-cols-1 md:grid-cols-2',
        '4' => 'grid-cols-1 sm:grid-cols-2 lg:grid-cols-4',
        default => 'grid-cols-1 md:grid-cols-2 lg:grid-cols-3',
    };

    // Global Currency Logic
    $siteSettings = app(\App\Settings\SiteSettings::class);
    $currencyType = $siteSettings->currency_type ?? 'text';
    $currencyText = $siteSettings->currency_text ?? 'ريال';
    $currencySvg = $siteSettings->currency_svg ?? '';
    $currencyImage = $siteSettings->currency_image ?? '';
    $currencyPos = $siteSettings->currency_position ?? 'after';
    $currencyIconSize = $siteSettings->currency_icon_size ?? 18;
    $priceTagBg = $siteSettings->price_tag_bg ?? '#FFFFFF';
    $priceColor = $siteSettings->price_tag_color ?? '#000000';

    $currencyHtml = '';
    if ($currencyType === 'text') {
        $currencyHtml = '<span class="text-sm font-bold opacity-90">' . e($currencyText) . '</span>';
    } elseif ($currencyType === 'svg') {
        $currencyHtml = '<span class="inline-block align-middle fill-current" style="width: ' . $currencyIconSize . 'px; height: ' . $currencyIconSize . 'px;">' . $currencySvg . '</span>';
    } elseif ($currencyType === 'image' && $currencyImage) {
        $currencyHtml = '<img src="' . asset('storage/' . $currencyImage) . '" class="inline-block align-middle object-contain" style="height: ' . $currencyIconSize . 'px; width: auto; max-width: none;" alt="currency">';
    }
@endphp

<style>
    .currency-container { display: flex; align-items: center; gap: 4px; }
</style>

<section class="py-20 bg-gray-50">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {{-- Section Header --}}
        <div class="text-center mb-12">
            <h2 class="text-3xl md:text-4xl font-bold text-gray-900 mb-4">{{ $title }}</h2>
            @if($description)
                <p class="text-lg text-gray-600 max-w-2xl mx-auto">{{ $description }}</p>
            @endif
            <div class="w-24 h-1 bg-theme-primary mx-auto mt-6 rounded-full"></div>
        </div>
        
        {{-- Packages Grid --}}
        @if($packages->count() > 0)
            <div class="grid {{ $gridClass }} gap-8">
                @foreach($packages as $package)
                    <div class="group bg-white rounded-2xl shadow-lg hover:shadow-2xl transition-all duration-300 overflow-hidden border border-gray-100 hover:-translate-y-2">
                        {{-- Image --}}
                        <div class="relative h-48 overflow-hidden">
                            @if($package->image)
                                <img src="{{ asset('storage/' . $package->image) }}" 
                                     alt="{{ $package->name }}"
                                     class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500">
                            @else
                                <div class="w-full h-full bg-gradient-to-br from-primary-100 to-primary-200 flex items-center justify-center">
                                    <svg class="w-16 h-16 text-theme-primary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M19.428 15.428a2 2 0 00-1.022-.547l-2.384-.477a6 6 0 00-3.86.517l-.318.158a6 6 0 01-3.86.517L6.05 15.21a2 2 0 00-1.806.547M8 4h8l-1 1v5.172a2 2 0 00.586 1.414l5 5c1.26 1.26.367 3.414-1.415 3.414H4.828c-1.782 0-2.674-2.154-1.414-3.414l5-5A2 2 0 009 10.172V5L8 4z"></path>
                                    </svg>
                                </div>
                            @endif
                            @if($package->sale_price)
                                <div class="absolute top-4 left-4 bg-red-500 text-white px-3 py-1 rounded-full text-sm font-bold">
                                    -{{ $package->discount_percentage }}%
                                </div>
                            @endif
                            @if($package->is_featured)
                                <div class="absolute top-4 right-4 bg-amber-400 text-amber-900 px-3 py-1 rounded-full text-xs font-bold">
                                    Featured
                                </div>
                            @endif
                        </div>
                        
                        {{-- Content --}}
                        <div class="p-6">
                            <h3 class="text-xl font-bold text-gray-900 mb-2 group-hover:text-theme-primary transition-colors">
                                {{ $package->name }}
                            </h3>
                            <p class="text-gray-600 text-sm mb-4 line-clamp-2">
                                {{ $package->short_description ?? Str::limit(strip_tags($package->description), 100) }}
                            </p>
                            
                            {{-- Features Preview --}}
                            @if($package->features && count($package->features) > 0)
                                <div class="mb-4 space-y-1">
                                    @foreach(array_slice($package->features, 0, 3) as $feature)
                                        <div class="flex items-center text-sm text-gray-600">
                                            <svg class="w-4 h-4 text-green-500 mr-2 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                                            </svg>
                                            {{ is_array($feature) ? ($feature['feature'] ?? '') : $feature }}
                                        </div>
                                    @endforeach
                                    @if(count($package->features) > 3)
                                        <p class="text-xs text-gray-500 pl-6">+{{ count($package->features) - 3 }} more</p>
                                    @endif
                                </div>
                            @endif
                            
                            {{-- Price --}}
                                <div class="flex items-center">
                                    @if($package->sale_price)
                                        <div class="flex flex-col">
                                            <div class="flex items-center text-gray-400 line-through text-xs gap-1">
                                                @if($currencyPos === 'before') {!! $currencyHtml !!} @endif
                                                {{ number_format($package->price) }}
                                                @if($currencyPos === 'after') {!! $currencyHtml !!} @endif
                                            </div>
                                            <div class="flex items-center text-xl font-bold gap-1 px-4 py-2 rounded-xl shadow-sm border border-black/5" style="color: {{ $priceColor }}; background: {{ $priceTagBg }};">
                                                @if($currencyPos === 'before') {!! $currencyHtml !!} @endif
                                                {{ number_format($package->sale_price) }}
                                                @if($currencyPos === 'after') {!! $currencyHtml !!} @endif
                                            </div>
                                        </div>
                                    @else
                                        <div class="flex items-center text-xl font-bold gap-1 px-4 py-2 rounded-xl shadow-sm border border-black/5" style="color: {{ $priceColor }}; background: {{ $priceTagBg }};">
                                            @if($currencyPos === 'before') {!! $currencyHtml !!} @endif
                                            {{ number_format($package->price) }}
                                            @if($currencyPos === 'after') {!! $currencyHtml !!} @endif
                                        </div>
                                    @endif
                                </div>
                                <a href="{{ route('package.show', $package->slug) }}" 
                                   class="px-4 py-2 bg-gray-900 text-white rounded-lg hover:bg-theme-primary transition-colors text-sm font-medium">
                                    View Details
                                </a>
                            </div>
                        </div>
                    </div>
                @endforeach
            </div>
            
            {{-- View All Button --}}
            @if($show_view_all)
                <div class="text-center mt-12">
                    <a href="{{ $view_all_url }}" 
                       class="inline-flex items-center px-8 py-4 bg-theme-primary text-white font-semibold rounded-lg hover:bg-theme-primary-dark transition-all transform hover:scale-105 shadow-lg shadow-primary-500/30">
                        {{ $view_all_text }}
                        <svg class="w-5 h-5 ml-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 8l4 4m0 0l-4 4m4-4H3"></path>
                        </svg>
                    </a>
                </div>
            @endif
        @else
            <div class="text-center py-12">
                <svg class="w-16 h-16 text-gray-300 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"></path>
                </svg>
                <p class="text-gray-500">No packages available at the moment.</p>
            </div>
        @endif
    </div>
</section>
