<?php

namespace App\Filament\Pages;

use App\Settings\SiteSettings;
use Filament\Actions\Action;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Grid;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Pages\Page;

class ManageSiteSettings extends Page implements HasForms
{
    use InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-cog-6-tooth';
    protected static ?string $navigationLabel = 'إعدادات الموقع';
    protected static ?string $title = 'إعدادات الموقع';
    protected static ?string $navigationGroup = 'الإعدادات';
    protected static ?int $navigationSort = 1;
    protected static string $view = 'filament.pages.manage-site-settings';

    public ?array $data = [];

    public function mount(): void
    {
        $settings = app(SiteSettings::class);
        $this->form->fill((array) $settings->toArray());
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Tabs::make('Settings')
                    ->tabs([
                        // Branding Tab
                        Tabs\Tab::make('الهوية والعلامة التجارية')
                            ->icon('heroicon-o-photo')
                            ->schema([
                                TextInput::make('site_name')
                                    ->label('اسم الموقع')
                                    ->required(),
                                
                                FileUpload::make('logo_light')
                                    ->label('الشعار (للخلفيات الفاتحة)')
                                    ->image()
                                    ->directory('logos')
                                    ->maxSize(2048),
                                
                                FileUpload::make('logo_dark')
                                    ->label('الشعار (للخلفيات الداكنة)')
                                    ->image()
                                    ->directory('logos')
                                    ->maxSize(2048),
                                
                                FileUpload::make('favicon')
                                    ->label('أيقونة الموقع (Favicon)')
                                    ->image()
                                    ->directory('favicons')
                                    ->maxSize(1024),
                            ]),

                        // Colors Tab
                        Tabs\Tab::make('ألوان التصميم')
                            ->icon('heroicon-o-swatch')
                            ->schema([
                                Grid::make(3)
                                    ->schema([
                                        ColorPicker::make('color_primary')
                                            ->label('اللون الأساسي')
                                            ->required(),
                                        
                                        ColorPicker::make('color_secondary')
                                            ->label('اللون الثانوي')
                                            ->required(),
                                        
                                        ColorPicker::make('color_accent')
                                            ->label('لون التمييز')
                                            ->required(),
                                    ]),
                                
                                Grid::make(3)
                                    ->schema([
                                        ColorPicker::make('color_light')
                                            ->label('لون الخلفية الفاتحة')
                                            ->required(),
                                        
                                        ColorPicker::make('color_text')
                                            ->label('لون النص')
                                            ->required(),
                                        
                                        ColorPicker::make('footer_bg')
                                            ->label('خلفية الفوتر')
                                            ->required(),
                                    ]),

                                Section::make('شريط التنقل (Navbar)')
                                    ->description('تحكم في مظهر وخلفية شريط التنقل العلوي')
                                    ->schema([
                                        Grid::make(2)
                                            ->schema([
                                                Select::make('navbar_theme')
                                                    ->label('مظهر الشريط العلوي')
                                                    ->options([
                                                        'dark'  => '🌙 داكن (Dark) — يستخدم شعار الخلفية الفاتحة',
                                                        'light' => '☀️ فاتح (Light) — يستخدم شعار الخلفية الداكنة',
                                                    ])
                                                    ->default('dark')
                                                    ->required()
                                                    ->helperText('اختر "داكن" إذا كانت خلفية الشريط داكنة، و"فاتح" إذا كانت بيضاء'),

                                                ColorPicker::make('navbar_bg_color')
                                                    ->label('لون خلفية الشريط')
                                                    ->helperText('يُطبَّق فقط عند اختيار المظهر الداكن'),
                                            ]),
                                    ]),
                            ]),

                        // Typography Tab
                        Tabs\Tab::make('الخطوط والاتجاه')
                            ->icon('heroicon-o-pencil-square')
                            ->schema([
                                Select::make('font_family')
                                    ->label('خط الموقع')
                                    ->options([
                                        'Cairo' => 'Cairo (عربي)',
                                        'Tajawal' => 'Tajawal (عربي)',
                                        'IBM Plex Sans' => 'IBM Plex Sans',
                                        'Inter' => 'Inter',
                                        'Roboto' => 'Roboto',
                                    ])
                                    ->default('Cairo')
                                    ->required(),
                                
                                Select::make('font_heading')
                                    ->label('خط العناوين')
                                    ->options([
                                        'Cairo' => 'Cairo (عربي)',
                                        'Tajawal' => 'Tajawal (عربي)',
                                        'IBM Plex Sans' => 'IBM Plex Sans',
                                        'Inter' => 'Inter',
                                    ])
                                    ->default('Cairo')
                                    ->required(),
                                
                                Select::make('direction')
                                    ->label('اتجاه الموقع')
                                    ->options([
                                        'rtl' => 'من اليمين لليسار (RTL)',
                                        'ltr' => 'من اليسار لليمين (LTR)',
                                    ])
                                    ->default('rtl')
                                    ->required(),
                                
                                TextInput::make('base_size')
                                    ->label('حجم الخط الأساسي (px)')
                                    ->numeric()
                                    ->default(16)
                                    ->required(),
                            ]),

                        // Contact Tab
                        Tabs\Tab::make('بيانات التواصل')
                            ->icon('heroicon-o-phone')
                            ->schema([
                                TextInput::make('contact_phone')
                                    ->label('رقم الهاتف')
                                    ->tel(),
                                
                                TextInput::make('contact_email')
                                    ->label('البريد الإلكتروني')
                                    ->email(),
                                
                                Textarea::make('contact_address')
                                    ->label('العنوان')
                                    ->rows(2),
                                
                                TextInput::make('contact_whatsapp')
                                    ->label('رقم الواتساب')
                                    ->placeholder('966500000000'),
                                
                                TextInput::make('working_hours')
                                    ->label('ساعات العمل')
                                    ->placeholder('24 ساعة'),
                                
                                TextInput::make('copyright_text')
                                    ->label('نص حقوق النشر')
                                    ->placeholder('© 2024 مختبرات بريزما. جميع الحقوق محفوظة.'),
                            ]),

                        // Social Links Tab
                        Tabs\Tab::make('مواقع التواصل')
                            ->icon('heroicon-o-share')
                            ->schema([
                                Grid::make(2)
                                    ->schema([
                                        TextInput::make('social_facebook')
                                            ->label('فيسبوك')
                                            ->url()
                                            ->prefixIcon('heroicon-m-link'),
                                        
                                        TextInput::make('social_twitter')
                                            ->label('تويتر / X')
                                            ->url()
                                            ->prefixIcon('heroicon-m-link'),
                                        
                                        TextInput::make('social_instagram')
                                            ->label('إنستغرام')
                                            ->url()
                                            ->prefixIcon('heroicon-m-link'),
                                        
                                        TextInput::make('social_linkedin')
                                            ->label('لينكدإن')
                                            ->url()
                                            ->prefixIcon('heroicon-m-link'),
                                        
                                        TextInput::make('social_youtube')
                                            ->label('يوتيوب')
                                            ->url()
                                            ->prefixIcon('heroicon-m-link'),
                                        
                                        TextInput::make('social_snapchat')
                                            ->label('سناب شات')
                                            ->url()
                                            ->prefixIcon('heroicon-m-link'),
                                        
                                        TextInput::make('social_tiktok')
                                            ->label('تيك توك')
                                            ->url()
                                            ->prefixIcon('heroicon-m-link'),
                                    ]),
                            ]),

                        // SEO Tab
                        Tabs\Tab::make('SEO والتحليلات')
                            ->icon('heroicon-o-globe-alt')
                            ->schema([
                                TextInput::make('meta_title')
                                    ->label('عنوان الموقع (Meta Title)')
                                    ->maxLength(70),
                                
                                Textarea::make('meta_description')
                                    ->label('وصف الموقع (Meta Description)')
                                    ->rows(3)
                                    ->maxLength(160),
                                
                                TagsInput::make('meta_keywords')
                                    ->label('الكلمات المفتاحية')
                                    ->separator(',')
                                    ->afterStateHydrated(function ($component, $state) {
                                        if (is_string($state) && $state !== '') {
                                            $component->state(array_map('trim', explode(',', $state)));
                                        } elseif (! is_array($state)) {
                                            $component->state([]);
                                        }
                                    })
                                    ->dehydrateStateUsing(fn ($state) => is_array($state) ? implode(', ', $state) : $state),
                                
                                Textarea::make('google_analytics')
                                    ->label('كود Google Analytics')
                                    ->rows(3)
                                    ->placeholder('أدخل كود التتبع هنا...'),
                                
                                Textarea::make('facebook_pixel')
                                    ->label('كود Facebook Pixel')
                                    ->rows(3)
                                    ->placeholder('أدخل كود البيكسل هنا...'),
                            ]),

                        // Booking & Packages Tab
                        Tabs\Tab::make('إعدادات الحجز والباقات')
                            ->icon('heroicon-o-shopping-cart')
                            ->schema([
                                Section::make('التحكم في أزرار الحجز')
                                    ->description('تفعيل أو إخفاء أزرار الحجز في صفحة الباقة')
                                    ->schema([
                                        Grid::make(4)
                                            ->schema([
                                                Toggle::make('show_booking_form')
                                                    ->label('نموذج الحجز')
                                                    ->helperText('إظهار نموذج الاسم والجوال'),
                                                
                                                Toggle::make('show_whatsapp_booking')
                                                    ->label('زر الواتساب')
                                                    ->helperText('إظهار زر الحجز عبر واتساب'),
                                                
                                                Toggle::make('show_credit_payment')
                                                    ->label('زر الدفع الإلكتروني')
                                                    ->helperText('إظهار زر الدفع بالبطاقة/تمارا/تابي'),
                                                
                                                Toggle::make('show_installment_note')
                                                    ->label('ملاحظة التقسيط')
                                                    ->helperText('إظهار نص التقسيط أسفل الأزرار'),
                                            ]),
                                    ]),
                                
                                Section::make('نصوص الأزرار')
                                    ->schema([
                                        Grid::make(3)
                                            ->schema([
                                                TextInput::make('booking_btn_text')
                                                    ->label('نص زر الحجز')
                                                    ->placeholder('احجز مجاناً الآن'),
                                                
                                                TextInput::make('whatsapp_btn_text')
                                                    ->label('نص زر الواتساب')
                                                    ->placeholder('احجز عبر واتساب'),
                                                
                                                TextInput::make('credit_btn_text')
                                                    ->label('نص زر الدفع')
                                                    ->placeholder('احجز بالدفع (تمارا / تابي / بطاقة)'),
                                            ]),
                                    ])
                                    ->collapsible(),
                                
                                Section::make('نموذج الحجز')
                                    ->description('تعديل تسميات حقول نموذج الحجز')
                                    ->schema([
                                        Grid::make(2)
                                            ->schema([
                                                TextInput::make('booking_form_name_label')
                                                    ->label('تسمية حقل الاسم')
                                                    ->placeholder('الاسم (مطلوب)'),
                                                
                                                TextInput::make('booking_form_name_placeholder')
                                                    ->label('نص مساعد لحقل الاسم')
                                                    ->placeholder('أدخل اسمك'),
                                                
                                                TextInput::make('booking_form_phone_label')
                                                    ->label('تسمية حقل الجوال')
                                                    ->placeholder('رقم الجوال (مطلوب)'),
                                                
                                                TextInput::make('booking_form_phone_placeholder')
                                                    ->label('نص مساعد لحقل الجوال')
                                                    ->placeholder('05xxxxxxxx'),
                                                
                                                TextInput::make('booking_form_note_label')
                                                    ->label('تسمية حقل الملاحظة')
                                                    ->placeholder('ملاحظة (اختياري)'),
                                                
                                                TextInput::make('booking_form_note_placeholder')
                                                    ->label('نص مساعد لحقل الملاحظة')
                                                    ->placeholder('أي ملاحظات إضافية...'),
                                            ]),
                                    ])
                                    ->collapsible(),
                                
                                Section::make('رسالة الواتساب والإشعارات')
                                    ->schema([
                                        Textarea::make('whatsapp_message_template')
                                            ->label('قالب رسالة الواتساب')
                                            ->helperText('استخدم {package} لاسم الباقة، {name} للاسم، {phone} للجوال')
                                            ->rows(4)
                                            ->placeholder("مرحباً، أود الحجز:\nالباقة: {package}\nالاسم: {name}\nالجوال: {phone}"),
                                        
                                        TextInput::make('booking_success_message')
                                            ->label('رسالة نجاح الحجز')
                                            ->placeholder('تم إرسال طلبك بنجاح! سنتواصل معك قريباً.'),
                                        
                                        TextInput::make('installment_note_text')
                                            ->label('نص ملاحظة التقسيط (عربي)')
                                            ->placeholder('قسّط مبلغ الباقة على 4 أقساط بدون فوائد عبر تمارا أو تابي'),
                                        
                                        TextInput::make('installment_note_text_en')
                                            ->label('نص ملاحظة التقسيط (إنجليزي)')
                                            ->placeholder('Split your payment into 4 interest-free installments via Tamara or Tabby'),
                                    ])
                                    ->collapsible(),
                                
                                Section::make('عناوين صفحة الباقات')
                                    ->schema([
                                        TextInput::make('package_page_title')
                                            ->label('عنوان صفحة الباقات')
                                            ->placeholder('باقات الفحوصات الطبية'),
                                        
                                        Textarea::make('package_page_subtitle')
                                            ->label('النص الفرعي لصفحة الباقات')
                                            ->rows(2)
                                            ->placeholder('باقات فحص شاملة بأفضل الأسعار...'),
                                    ])
                                    ->collapsible(),
                            ]),
                    ])
                    ->columnSpanFull(),
            ])
            ->statePath('data');
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('save')
                ->label('حفظ الإعدادات')
                ->submit('save')
                ->icon('heroicon-m-check'),
        ];
    }

    public function save(): void
    {
        $data = $this->form->getState();
        
        $settings = app(SiteSettings::class);
        
        foreach ($data as $key => $value) {
            $settings->$key = $value;
        }
        
        $settings->save();
        
        Notification::make()
            ->title('تم حفظ الإعدادات بنجاح')
            ->success()
            ->send();
    }
}
