<?php

namespace App\Livewire;

use App\Models\Page;
use Livewire\Component;
use Livewire\WithFileUploads;

class LiveBuilder extends Component
{
    use WithFileUploads;

    public Page $page;
    public array $content = [];
    public array $settings = [];
    public string $activeDevice = 'desktop'; // desktop, tablet, mobile
    public string $previewLocale = 'ar';     // ar or en
    public ?int $activeBlockIndex = null;

    public function mount(Page $page)
    {
        $this->page = $page;
        $this->content = $page->content ?? [];
        $this->settings = [
            'primary_color' => '#00A0B0',
            'secondary_color' => '#2D9CDB',
            'bg_texture' => true,
        ];
    }

    public function selectBlock(?int $index): void
    {
        $this->activeBlockIndex = $index;
    }

    public function getAvailableBlocks()
    {
        return [
            ['type' => 'prisma_hero', 'label' => 'Hero Section', 'icon' => 'heroicon-o-home'],
            ['type' => 'prisma_categories', 'label' => 'Categories', 'icon' => 'heroicon-o-squares-2x2'],
            ['type' => 'prisma_packages', 'label' => 'Packages Carousel', 'icon' => 'heroicon-o-shopping-bag'],
            ['type' => 'prisma_about', 'label' => 'About Section', 'icon' => 'heroicon-o-information-circle'],
            ['type' => 'prisma_services', 'label' => 'Our Services', 'icon' => 'heroicon-o-beaker'],
            ['type' => 'prisma_faq', 'label' => 'FAQ Section', 'icon' => 'heroicon-o-question-mark-circle'],
            ['type' => 'prisma_cta', 'label' => 'Call to Action', 'icon' => 'heroicon-o-megaphone'],
            ['type' => 'custom_form', 'label' => 'Custom Form', 'icon' => 'heroicon-o-document-text'],
        ];
    }

    public function addBlock($type)
    {
        $newBlock = [
            'type' => $type,
            'data' => $this->getDefaultDataForBlock($type)
        ];
        
        $this->content[] = $newBlock;
        $this->activeBlockIndex = count($this->content) - 1;
    }

    public function removeBlock($index)
    {
        unset($this->content[$index]);
        $this->content = array_values($this->content); // Re-index
        if ($this->activeBlockIndex === $index) {
            $this->activeBlockIndex = null;
        } elseif ($this->activeBlockIndex !== null && $this->activeBlockIndex > $index) {
            $this->activeBlockIndex--;
        }
    }

    public function moveBlock($index, $direction)
    {
        $targetIndex = $direction === 'up' ? $index - 1 : $index + 1;
        
        if ($targetIndex >= 0 && $targetIndex < count($this->content)) {
            $temp = $this->content[$index];
            $this->content[$index] = $this->content[$targetIndex];
            $this->content[$targetIndex] = $temp;
            $this->activeBlockIndex = $targetIndex;
        }
    }

    protected function getDefaultDataForBlock($type)
    {
        return match($type) {
            'prisma_hero' => [
                'layout_design' => 'premium',
                'premium_display_mode' => 'full',
                'show_background_image' => true,
                'show_hero_text' => true,
                'show_cta_button' => true,
                'show_categories' => true,
                'title' => 'اختر باقتين والثالثة مجاناً',
                'title_ar' => 'اختر باقتين والثالثة مجاناً',
                'title_en' => 'Pick Two, Get the Third Free',
                'subtitle' => 'تطمن على صحتك ولا يوقفك برد الشتاء',
                'subtitle_ar' => 'تطمن على صحتك ولا يوقفك برد الشتاء',
                'subtitle_en' => 'Take care of your health this winter',
                'cta_text' => 'احجز الآن',
                'cta_text_ar' => 'احجز الآن',
                'cta_text_en' => 'Book Now',
                'cta_url' => '#packages',
                'button_style' => 'primary',
                'cta_bg_start' => '#0099FF',
                'cta_bg_end' => '#7B00FF',
                'cta_text_color' => '#ffffff',
                'background_image' => null,
                'image_has_text' => false,
                'image_position' => 'right',
                'image_height' => 'large',
                'bg_color' => '#0a0a0a',
                'overlay_color' => '#000000',
                'overlay_opacity' => 30,
                'text_color' => '#ffffff',
                'show_pattern' => true,
                'full_width' => true,
                'rounded_corners' => false,
                'add_shadow' => false,
                'categories' => [
                    ['name' => 'الفحوصات الشاملة', 'icon' => 'fas fa-vial', 'url' => '#', 'use_image' => false, 'image' => null],
                    ['name' => 'باقة صحة الرجل', 'icon' => 'fas fa-stethoscope', 'url' => '#', 'use_image' => false, 'image' => null],
                    ['name' => 'باقة صحة المرأة', 'icon' => 'fas fa-heart', 'url' => '#', 'use_image' => false, 'image' => null],
                    ['name' => 'باقة الطفل', 'icon' => 'fas fa-baby', 'url' => '#', 'use_image' => false, 'image' => null],
                ],
            ],
            'prisma_categories' => [
                'show_from_database' => true,
                'limit' => 8,
                'show_names' => true,
                'card_style' => true,
            ],
            'prisma_packages' => [
                'title' => 'باقات مختبرات بريزما',
                'title_ar' => 'باقات مختبرات بريزما',
                'title_en' => 'Prisma Lab Packages',
                'subtitle' => 'عروض حصرية لفترة محدودة',
                'subtitle_ar' => 'عروض حصرية لفترة محدودة',
                'subtitle_en' => 'Exclusive offers for a limited time',
                'badge_text' => 'عروض الموسم المميزة',
                'show_tier_99' => true,
                'show_tier_199' => true,
                'show_tier_299' => true,
                'show_description' => true,
                'show_features' => true,
                'show_badge' => true,
                'show_tests_count' => true,
                'show_old_price' => true,
                'card_hover_effect' => true,
            ],
            'prisma_about' => [
                'title' => 'مختبرات بريزما الطبية – شريكك الصحي الموثوق',
                'title_ar' => 'مختبرات بريزما الطبية – شريكك الصحي الموثوق',
                'title_en' => 'Prisma Labs — Your Trusted Health Partner',
                'description' => 'مختبرات بريزما الطبية هي مختبر تحاليل معتمد يقدّم خدمات مخبرية متكاملة.',
                'description_ar' => 'مختبرات بريزما الطبية هي مختبر تحاليل معتمد يقدّم خدمات مخبرية متكاملة.',
                'description_en' => 'Prisma Medical Labs is an accredited analysis laboratory offering comprehensive services.',
                'button_text' => 'اكتشف الباقات',
                'button_url' => '/packages',
                'show_call_button' => true,
                'show_default_icon' => true,
                'show_stats' => true,
                'image_on_right' => false,
                'full_width' => true,
            ],
            'prisma_services' => [
                'title' => 'خدماتنا المخبرية',
                'title_ar' => 'خدماتنا المخبرية',
                'title_en' => 'Our Lab Services',
                'subtitle' => 'تقنيات متطورة لنتائج أدق',
                'subtitle_ar' => 'تقنيات متطورة لنتائج أدق',
                'subtitle_en' => 'Advanced technology for accurate results',
                'show_icon_background' => true,
                'card_border' => true,
                'center_align' => true,
            ],
            'prisma_faq' => [
                'title' => 'الأسئلة الشائعة حول خدماتنا',
                'title_ar' => 'الأسئلة الشائعة حول خدماتنا',
                'title_en' => 'Frequently Asked Questions',
                'show_icon' => true,
                'open_first' => true,
                'allow_multiple_open' => false,
                'full_width' => true,
                'faqs' => [
                    ['question' => 'هل التقارير معتمدة؟', 'answer' => 'نعم، جميع تقاريرنا معتمدة من وزارة الصحة السعودية وتُرسل مباشرة إلى تطبيق صحتي.'],
                    ['question' => 'متى تظهر النتائج؟', 'answer' => 'تظهر معظم النتائج خلال 24 ساعة من وقت سحب العينة.'],
                    ['question' => 'هل تتوفر خدمة السحب المنزلي؟', 'answer' => 'نعم، نوفر خدمة السحب المنزلي المجانية داخل مدينة الرياض.'],
                ],
            ],
            'prisma_cta' => [
                'title' => 'جاهز لبدء رحلة الاطمئنان؟',
                'title_ar' => 'جاهز لبدء رحلة الاطمئنان؟',
                'title_en' => 'Ready to start your wellness journey?',
                'subtitle' => 'احجز موعدك الآن عبر الواتساب واحصل على خصم فوري',
                'subtitle_ar' => 'احجز موعدك الآن عبر الواتساب واحصل على خصم فوري',
                'subtitle_en' => 'Book your appointment now via WhatsApp and get an instant discount',
                'button_text' => 'احجز الآن',
                'bg_color_start' => '#1A365D',
                'bg_color_end' => '#00A0B0',
                'rounded_corners' => true,
                'full_width' => false,
                'margin_top' => true,
            ],
            'custom_form' => [
                'title' => 'تواصل معنا',
                'description' => 'يسعدنا تواصلكم واستقبال استفساراتكم.',
                'custom_form_id' => null,
            ],
            default => []
        };
    }

    public function setDevice($device)
    {
        $this->activeDevice = $device;
    }

    public function setPreviewLocale(string $locale): void
    {
        if (in_array($locale, ['ar', 'en'])) {
            $this->previewLocale = $locale;
        }
    }

    /**
     * Get the label for a block type.
     */
    public function getBlockLabel(string $type): string
    {
        $labels = [
            'prisma_hero' => 'بانر رئيسي',
            'prisma_hero_grid' => 'بانر شبكي',
            'prisma_categories' => 'التصنيفات',
            'prisma_packages' => 'الباقات',
            'prisma_about' => 'من نحن',
            'prisma_services' => 'الخدمات',
            'prisma_faq' => 'الأسئلة الشائعة',
            'prisma_cta' => 'دعوة للإجراء',
            'hero_slider' => 'سلايدر',
            'package_grid' => 'شبكة باقات',
            'features' => 'المميزات',
            'cta' => 'زر إجراء',
            'html' => 'HTML مخصص',
            'custom_form' => 'نموذج مخصص',
        ];

        return $labels[$type] ?? ucfirst(str_replace('_', ' ', $type));
    }

    /**
     * Get human-readable label for a data key.
     */
    public function getFieldLabel(string $key): string
    {
        $labels = [
            'title' => 'العنوان',
            'title_ar' => 'العنوان (عربي)',
            'title_en' => 'Title (English)',
            'subtitle' => 'العنوان الفرعي',
            'subtitle_ar' => 'العنوان الفرعي (عربي)',
            'subtitle_en' => 'Subtitle (English)',
            'description' => 'الوصف',
            'description_ar' => 'الوصف (عربي)',
            'description_en' => 'Description (English)',
            'cta_text' => 'نص الزر',
            'cta_text_ar' => 'نص الزر (عربي)',
            'cta_text_en' => 'Button Text (English)',
            'cta_url' => 'رابط الزر',
            'button_text' => 'نص الزر',
            'button_url' => 'رابط الزر',
            'bg_color' => 'لون الخلفية',
            'bg_color_start' => 'بداية التدرج',
            'bg_color_end' => 'نهاية التدرج',
            'text_color' => 'لون النص',
            'overlay_color' => 'لون الطبقة',
            'overlay_opacity' => 'شفافية الطبقة (%)',
            'cta_bg_start' => 'بداية تدرج الزر',
            'cta_bg_end' => 'نهاية تدرج الزر',
            'cta_text_color' => 'لون نص الزر',
            'layout_design' => 'تصميم القالب',
            'premium_display_mode' => 'وضع العرض المميز',
            'image_position' => 'موضع الصورة',
            'image_height' => 'ارتفاع الصورة',
            'button_style' => 'نمط الزر',
            'badge_text' => 'نص الشارة',
            'show_background_image' => 'إظهار صورة الخلفية',
            'show_hero_text' => 'إظهار النص',
            'show_cta_button' => 'إظهار زر CTA',
            'show_categories' => 'إظهار التصنيفات',
            'show_pattern' => 'إظهار النمط',
            'full_width' => 'عرض كامل',
            'rounded_corners' => 'حواف مستديرة',
            'add_shadow' => 'إضافة ظل',
            'image_has_text' => 'الصورة تحتوي نص',
            'show_from_database' => 'عرض من قاعدة البيانات',
            'show_names' => 'إظهار الأسماء',
            'card_style' => 'نمط البطاقة',
            'show_tier_99' => 'إظهار فئة 99',
            'show_tier_199' => 'إظهار فئة 199',
            'show_tier_299' => 'إظهار فئة 299',
            'show_description' => 'إظهار الوصف',
            'show_features' => 'إظهار المميزات',
            'show_badge' => 'إظهار الشارة',
            'show_tests_count' => 'إظهار عدد الفحوصات',
            'show_old_price' => 'إظهار السعر القديم',
            'card_hover_effect' => 'تأثير تحويم البطاقة',
            'show_call_button' => 'إظهار زر الاتصال',
            'show_default_icon' => 'إظهار الأيقونة',
            'show_stats' => 'إظهار الإحصائيات',
            'image_on_right' => 'الصورة يمين',
            'show_icon_background' => 'خلفية الأيقونة',
            'card_border' => 'حدود البطاقة',
            'center_align' => 'محاذاة وسط',
            'show_icon' => 'إظهار الأيقونة',
            'open_first' => 'فتح الأول',
            'allow_multiple_open' => 'فتح متعدد',
            'margin_top' => 'مسافة علوية',
            'limit' => 'الحد الأقصى',
        ];

        return $labels[$key] ?? ucfirst(str_replace('_', ' ', $key));
    }

    /**
     * Determine the input type for a field.
     */
    public function getFieldType(string $key, mixed $value): string
    {
        // Boolean fields
        if (is_bool($value)) {
            return 'toggle';
        }

        // Color fields
        if (str_contains($key, 'color') || str_contains($key, 'bg_color')) {
            return 'color';
        }

        // URL fields
        if (str_contains($key, 'url') || str_contains($key, '_url')) {
            return 'url';
        }

        // Select fields
        if (in_array($key, ['layout_design', 'premium_display_mode', 'button_style', 'image_position', 'image_height'])) {
            return 'select';
        }

        // Numeric fields
        if (is_numeric($value) && in_array($key, ['overlay_opacity', 'limit', 'sort_order'])) {
            return 'number';
        }

        // Complex arrays (categories, faqs, stats, etc.)
        if (is_array($value)) {
            return 'complex';
        }

        // Long text
        if (is_string($value) && strlen($value) > 100) {
            return 'textarea';
        }

        // Null/image fields
        if ($value === null && (str_contains($key, 'image') || str_contains($key, 'background'))) {
            return 'skip';
        }

        return 'text';
    }

    /**
     * Get options for select fields.
     */
    public function getSelectOptions(string $key): array
    {
        return match($key) {
            'layout_design' => [
                'premium' => 'Premium (Floating Cards)',
                'center_text_on_image' => 'Center Text on Image',
                'text_beside_image' => 'Text Beside Image',
                'text_below_image' => 'Text Below Image',
                'image_only' => 'Image Only',
                'text_only' => 'Text Only',
            ],
            'premium_display_mode' => [
                'full' => 'Image + Text + Categories',
                'image_only' => 'Image Only',
                'text_only' => 'Text Only',
            ],
            'button_style' => [
                'primary' => 'Primary (Gradient)',
                'secondary' => 'Secondary (Outline)',
                'white' => 'White',
                'ghost' => 'Ghost',
            ],
            'image_position' => [
                'left' => 'Left',
                'right' => 'Right',
            ],
            'image_height' => [
                'small' => 'Small (300px)',
                'medium' => 'Medium (450px)',
                'large' => 'Large (600px)',
                'fullscreen' => 'Full Screen',
            ],
            default => [],
        };
    }

    public function save()
    {
        $this->page->update([
            'content' => $this->content
        ]);

        $this->dispatch('page-saved');
        $this->dispatch('notify', [
            'type' => 'success',
            'message' => 'تم حفظ التغييرات بنجاح'
        ]);
    }

    public function render()
    {
        // Apply the preview locale so block views render in the selected language
        app()->setLocale($this->previewLocale);

        return view('livewire.live-builder')
            ->layout('components.layouts.builder');
    }
}
